// /api/discover/whitespace — White Space Score
// Uses ytGet() for automatic key rotation on 403/quota errors.

import { NextResponse } from 'next/server';
import { ytGet } from '../../../../lib/youtubeApi';

function clamp(v: number, min = 0, max = 100) {
    return Math.max(min, Math.min(max, v));
}

export async function GET(req: Request) {
    const { searchParams } = new URL(req.url);
    const niche = searchParams.get('niche');
    if (!niche) return NextResponse.json({ error: 'niche param required' }, { status: 400 });

    try {
        // 1. DEMAND — top videos (search.list = 100 units, rotated)
        const videoSearch = await ytGet('/search', {
            q: niche, type: 'video', part: 'id', order: 'viewCount', maxResults: 10,
        });
        const videoIds: string[] = (videoSearch.items || []).map((v: any) => v.id.videoId).filter(Boolean);
        const totalResults: number = videoSearch.pageInfo?.totalResults || 0;

        // Get actual view counts (videos.list = 1 unit)
        let avgViews = 0;
        if (videoIds.length) {
            const statsData = await ytGet('/videos', { id: videoIds.join(','), part: 'statistics' });
            const views = (statsData.items || []).map((v: any) => parseInt(v.statistics?.viewCount || '0'));
            avgViews = views.length ? views.reduce((a: number, b: number) => a + b, 0) / views.length : 0;
        }

        // 2. SUPPLY — channel count (search.list = 100 units, rotated)
        const channelSearch = await ytGet('/search', {
            q: niche, type: 'channel', part: 'id', maxResults: 5,
        });
        const channelCount: number = channelSearch.pageInfo?.totalResults || 0;

        // Top channel avg subs (channels.list = 1 unit)
        const channelIds: string[] = (channelSearch.items || []).map((c: any) => c.id.channelId).filter(Boolean);
        let avgSubs = 0;
        if (channelIds.length) {
            const chData = await ytGet('/channels', { id: channelIds.join(','), part: 'statistics' });
            const subs = (chData.items || []).map((c: any) => parseInt(c.statistics?.subscriberCount || '0'));
            avgSubs = subs.length ? subs.reduce((a: number, b: number) => a + b, 0) / subs.length : 0;
        }

        // 3. SCORE
        const demandRaw = Math.log10(Math.max(avgViews, 1)) / 7 * 100;
        const demandScore = clamp(demandRaw);
        const supplyRaw = Math.log10(Math.max(channelCount, 1)) / 6 * 100;
        const supplyScore = clamp(supplyRaw);
        const establishedPenalty = avgSubs > 500000 ? 20 : avgSubs > 100000 ? 10 : 0;
        const whiteSpaceScore = clamp(demandScore - supplyScore + 50 - establishedPenalty);

        const label =
            whiteSpaceScore >= 75 ? '🟢 Prime White Space' :
                whiteSpaceScore >= 55 ? '🟡 Moderate Opportunity' :
                    whiteSpaceScore >= 35 ? '🟠 Competitive' : '🔴 Graveyard';

        const verdict =
            whiteSpaceScore >= 75 ? 'High demand with very few active channels — excellent entry opportunity.' :
                whiteSpaceScore >= 55 ? 'Decent demand but growing competition. Move fast.' :
                    whiteSpaceScore >= 35 ? 'Well-established niche. Differentiation required to compete.' :
                        'Over-saturated with low demand signals. Avoid unless you have a unique angle.';

        return NextResponse.json({
            niche,
            whiteSpaceScore: Math.round(whiteSpaceScore),
            demandScore: Math.round(demandScore),
            supplyScore: Math.round(supplyScore),
            label, verdict,
            data: {
                avgViews: Math.round(avgViews),
                totalVideoResults: totalResults,
                totalChannels: channelCount,
                avgTopChannelSubs: Math.round(avgSubs),
            },
        });

    } catch (err: any) {
        console.error('[WhiteSpace]', err);
        if (err?.quotaExhausted) {
            return NextResponse.json({ error: err.message, quotaExhausted: true }, { status: 503 });
        }
        return NextResponse.json({ error: err.message }, { status: 500 });
    }
}
